//******************************************************************************
//   MSP-XXX6 Demo - SMBus Master transmits & receives from SMBus Slave
//
//   Description: This demo connects two MSP430's via the I2C/SMBus. This slave
//   will receive byte transmissions without PEC. It will transmit 1 byte then
//   clock stretch by delaying the fill of the TX buffer and cause a timeout.
//   This is the SLAVE code.  This is used with msp430x5xx_SMB_MST_timeout.c 
//   to cause a timeout condition. This code can also be used to test normal 
//   transactions without timeout by commenting out one line. Please read the 
//   comment in the code below. LED on P1.0 blinks when a timeout happens.
//  
//   ACLK = 32kHz, MCLK = SMCLK = default DCO = ~1.045MHz
//   
//   
//                                 /|\  /|\
//                 MSP430F5438     10k  10k    MSP430F5438
//                    slave         |    |        master           
//              -----------------|  |    |  ----------------- 
//             |             P3.7|<-|----+>|P3.7         P1.0|-->LED
//             |                 |  |      |                 |
//             |             P5.4|<-+----->|P5.4             |
//             |                 |         |                 |
//             |             P2.0|<--------|P2.0             |
//
//
//  J. Kam / H. Grewal
//  Texas Instruments Inc.
//  July 2009
//  Built with Code Composer Essentials v3.1 Build: 3.2.4.3.8 
//******************************************************************************

#include "msp430x54x.h"

unsigned char data_received, count=0;
unsigned char RXData;                       // Recieve Dummy Byte on timeout

void i2c_init(void);                        

void main(void)
{
  WDTCTL = WDTPW + WDTHOLD;                 // Stop WDT

  i2c_init();                               // Initializes USCI for I2C
  P1DIR |= BIT0;                            // LED initialization
  P1OUT = 0;                           
  
  P2DIR &= ~BIT0;                           // P2.0 is reset cmd from master
  P2IES &= ~BIT0;                           // Low-to-high trans sets P2IFG
  P2IFG = 0;                                // Clear any P2IFG
  P2IE |= BIT0;                             // Enable P2.0 ISR 
  __bis_SR_register(LPM4_bits + GIE);       // Enter LPM4, enable interrupts                                        
} 

void i2c_init(void)
{
  P5SEL |= BIT4 + BIT5;                     // Assigns I2C pins on 5438
  P3SEL |= BIT7;
  P3SEL &= ~BIT6;
  P3DIR &= ~BIT6;                           // Create a slave select line
  UCB1CTL1 |= UCSWRST;                      // Enable SW reset
  UCB1CTL0 = UCMODE_3 + UCSYNC;             // I2C Slave, synchronous mode
  UCB1I2COA = 0x48;                         // Own address is 0x48
  UCB1CTL1 &= ~UCSWRST;                     // Clear SW reset, resume operation
  UCB1IE |= UCRXIE + UCTXIE;
}
/*----------------------------------------------------------------------------+
| I2C/SMBus interrupt (USCI B1)
+----------------------------------------------------------------------------*/
#pragma vector = USCI_B1_VECTOR
__interrupt void USCI_B1_ISR(void)
{
  switch(__even_in_range(UCB1IV, 12))
  {
    case 0:
      break;
    case 2:                                 // ALIFG
      break;
    case 4:                                 // NACKIFG
      break;
    case 6:                                 // STTIFG
      break;
    case 8:                                 // STPIFG
      break;
    case 10:  data_received = UCB1RXBUF;    // RXIFG
      break; 
    case 12:  if(count==0)                  // TXIFG
              {
                UCB1IE &= ~UCTXIE;          // Read the comment below
      /* Comment the line above if normal Slave operation is desired where
      2 bytes are Transmitted by Slave without the timeout condition.
      If the above line is left uncommented, the 2nd byte will NOT be
      transmitted simulating a delay by the slave to hold the SCL line low and 
      a timeout will occur leading to a port interrupt */
                UCB1TXBUF = ++data_received;  // TX Data 
                count++;
              }
              else
              {
                __delay_cycles(200);        // Normal delay between bytes
                UCB1TXBUF = ++data_received;// TX data
                count = 0;
              }
              
      break;
    default:
      break;
  }
}


#pragma vector=PORT2_VECTOR
__interrupt void PORT2_VECTOR_ISR(void)
{
  UCB1IE &= ~(UCRXIE + UCTXIE);             // Disable Interrupts
  UCB1IFG &= ~(UCTXIFG + UCRXIFG);          // Clear Flags
  P2IFG = 0;                                // Manually Clear port interrupt
  UCB1TXBUF = 0;                            // TX Dummy data to release line
  RXData = UCB1RXBUF;                       // RX Dummy data to release line
  P1OUT ^= 0x01;                            // Toggle LED
  i2c_init();                               // Reset Slave
  count = 0;                                // Reset the counter for 1st byte TX
}
